/*----------------------------------------------------------------------------
 * avs2bdnxml - Generates BluRay subtitle stuff from RGBA AviSynth scripts
 * Copyright (C) 2008-2009 Arne Bochem <avs2bdnxml at ps-auxw de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *----------------------------------------------------------------------------
 * Version 1.9
 *   - Correct calculation of timecodes. Integer only, now.
 *   - Remove support for drop timecodes. If anybody really needs them, and
 *     has a way to check calculated timecodes are correct, please tell me.
 *
 * Version 1.8
 *   - Fix crash with certain frame sizes (unaligned buffer used with SSE2)
 *
 * Version 1.7
 *   - Terminate using exit/return instead of abort (no more "crashes")
 *
 * Version 1.6
 *   - Zero transparent pixels to guard against invisible color information
 *     (Slight slowdown, but more robustness)
 *   - Add SSE2 runtime detection
 *   - Correctly set DropFrame attribute
 *
 * Version 1.5
 *   - Validate framerates
 *   - Make non-drop timecodes the default for 23.976, 29.97 and 59.94
 *   - Fix bad timecode calculation due to float math optimization
 *   - Add integer SSE2 optimization (but most time is spent in AviSynth)
 *   - Don't uselessly swap R and B channels
 *
 * Version 1.4
 *   - Add 30 to framerate list
 *   - Enhance timecode precision
 *
 * Version 1.3
 *   - Read frames correctly in avs2bdnxml. No more FlipVertical() necessary.
 *
 * Version 1.2
 *   - Fixed bug, where only the bottom part of the image was checked for
 *     changes
 *
 * Version 1.1
 *   - Fixed crash bug with videos shorter than 1000 frames.
 *   - Convert from BGR to RGB, colors in PNGs should be correct now.
 *
 * Version 1.0
 *   - Initial release. Seems to work.
 *----------------------------------------------------------------------------
 * Thanks to:
 *   - Daiz in #x264 for inspiration, testing, and bug reports.
 *   - Loren Merritt and Laurent Aimar for writing x264's AVS reader code.
 *----------------------------------------------------------------------------*/

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <png.h>

/* AVIS input code taken from muxers.c from the x264 project (GPLv2 or later).
 * Authors: Laurent Aimar <fenrir@via.ecp.fr>
 *          Loren Merritt <lorenm@u.washington.edu>
 * Slightly modified, so if you find breakage, it's probably mine. :)
 */
#include <windows.h>
#include <vfw.h>

static int64_t gcd( int64_t a, int64_t b )
{
    while (1)
    {
        int64_t c = a % b;
        if( !c )
            return b;
        a = b;
        b = c;
    }
}

typedef struct {
    PAVISTREAM p_avi;
    int width, height;
} avis_input_t;

typedef struct {
    int i_width;
    int i_height;
    int i_fps_den;
    int i_fps_num;
} stream_info_t;

int open_file_avis( char *psz_filename, avis_input_t **p_handle, stream_info_t *p_param )
{
    avis_input_t *h = malloc(sizeof(avis_input_t));
    AVISTREAMINFO info;
    int i;

    *p_handle = h;

    AVIFileInit();
    if( AVIStreamOpenFromFile( &h->p_avi, psz_filename, streamtypeVIDEO, 0, OF_READ, NULL ) )
    {
        AVIFileExit();
        return -1;
    }

    if( AVIStreamInfo(h->p_avi, &info, sizeof(AVISTREAMINFO)) )
    {
        AVIStreamRelease(h->p_avi);
        AVIFileExit();
        return -1;
    }

    /* Check input format */
    if (info.fccHandler != MAKEFOURCC('D', 'I', 'B', ' '))
    {
        fprintf( stderr, "avis [error]: unsupported input format (%c%c%c%c)\n",
            (char)(info.fccHandler & 0xff), (char)((info.fccHandler >> 8) & 0xff),
            (char)((info.fccHandler >> 16) & 0xff), (char)((info.fccHandler >> 24)) );

        AVIStreamRelease(h->p_avi);
        AVIFileExit();

        return -1;
    }

    h->width =
    p_param->i_width = info.rcFrame.right - info.rcFrame.left;
    h->height =
    p_param->i_height = info.rcFrame.bottom - info.rcFrame.top;
    i = gcd(info.dwRate, info.dwScale);
    p_param->i_fps_den = info.dwScale / i;
    p_param->i_fps_num = info.dwRate / i;

    fprintf( stderr, "avis [info]: %dx%d @ %.2f fps (%d frames)\n",
        p_param->i_width, p_param->i_height,
        (double)p_param->i_fps_num / (double)p_param->i_fps_den,
        (int)info.dwLength );

    return 0;
}

int get_frame_total_avis( avis_input_t *handle )
{
    avis_input_t *h = handle;
    AVISTREAMINFO info;

    if( AVIStreamInfo(h->p_avi, &info, sizeof(AVISTREAMINFO)) )
        return -1;

    return info.dwLength;
}

int read_frame_avis( char *p_pic, avis_input_t *handle, int i_frame )
{
    avis_input_t *h = handle;

    if( AVIStreamRead(h->p_avi, i_frame, 1, p_pic, h->width * h->height * 4, NULL, NULL ) )
        return -1;

    return 0;
}

int close_file_avis( avis_input_t *handle )
{
    avis_input_t *h = handle;
    AVIStreamRelease(h->p_avi);
    AVIFileExit();
    free(h);
    return 0;
}

/* AVIS code ends here */

/* Main avs2bdnxml code starts here, too */

void write_png(int file_id, uint8_t *image, int w, int h)
{
	FILE *fh;
	png_structp png_ptr;
	png_infop info_ptr;
	png_bytep *row_pointers;
	char filename[13];
	int i;

	snprintf(filename, 13, "%08d.png", file_id);

	if ((fh = fopen(filename, "wb")) == NULL)
	{
		perror("Cannot open PNG file for writing");
		exit(1);
	}

	/* Initialize png struct */
	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (png_ptr == NULL)
	{
		fprintf(stderr, "Cannot create png_ptr.\n");
		exit(1);
	}

	/* Initialize info struct */
	info_ptr = png_create_info_struct(png_ptr);
	if (info_ptr == NULL)
	{
		png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
		fprintf(stderr, "Cannot create info_ptr.\n");
		exit(1);
	}

	/* Set long jump stuff (weird..?) */
	if (setjmp(png_jmpbuf(png_ptr)))
	{
		png_destroy_write_struct(&png_ptr, &info_ptr);
		fclose(fh);
		fprintf(stderr, "Error while writing PNG file: %s\n", filename);
		exit(1);
	}

	/* Initialize IO */
	png_init_io(png_ptr, fh);

	/* Set file info */
	png_set_IHDR(png_ptr, info_ptr, w, h, 8, PNG_COLOR_TYPE_RGB_ALPHA, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

	/* Allocate row pointer memory */
	row_pointers = calloc(h, sizeof(png_bytep));

	/* Set row pointers */
	for (i = 0; i < h; i++)
	{
		row_pointers[i] = image + i * w * 4;
	}
	png_set_rows(png_ptr, info_ptr, row_pointers);

	/* Set compression */
	png_set_filter(png_ptr, 0, PNG_FILTER_VALUE_SUB);
	png_set_compression_level(png_ptr, 5);

	/* Write image */
	png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);

	/* Free memory */
	png_destroy_write_struct(&png_ptr, &info_ptr);
	free(row_pointers);

	/* Close file handle */
	fclose(fh);
}

int is_identical_sse2 (stream_info_t *s_info, char *img, char *img_old)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;
	char zero[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	char a_mask[16] = {0,0,0,0xff,0,0,0,0xff,0,0,0,0xff,0,0,0,0xff};
	char full[16] = {0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff};
	int volatile different = 0;
	int volatile *diff_ptr = &different;

	asm __volatile__ ("movupd (%%eax), %%xmm3\n" : : "a" (zero));
	asm __volatile__ ("movupd (%%ecx), %%xmm4\n" : : "c" (a_mask));
	asm __volatile__ ("movupd (%%edx), %%xmm7\n" : : "d" (full));
	while (img < max && !different)
	{
		asm __volatile__
		(
			"movapd (%%eax), %%xmm0\n"
			"movapd %%xmm0, %%xmm5\n"
			"pand %%xmm4, %%xmm5\n"
			"movapd (%%esi), %%xmm1\n"
			"pcmpeqd %%xmm3, %%xmm5\n"
			"pxor %%xmm7, %%xmm5\n"
			"pand %%xmm5, %%xmm0\n"
			"movapd %%xmm0, (%%eax)\n"
			"psadbw %%xmm1, %%xmm0\n"
			"movd %%xmm0, (%%edx)\n"
			: "=a" (img), "=d" (diff_ptr)
			: "a" (img), "S" (img_old), "d" (diff_ptr)
		);
		img += 16;
		img_old += 16;
	}

	if (different)
		return 0;

	return 1;
}

int is_empty_sse2 (stream_info_t *s_info, char *img)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;
	char zero[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	char a_mask[16] = {0,0,0,0xff,0,0,0,0xff,0,0,0,0xff,0,0,0,0xff};
	int volatile not_empty = 0;
	int volatile *ne_ptr = &not_empty;

	asm __volatile__ ("movupd (%%eax), %%xmm1\n" : : "a" (zero));
	asm __volatile__ ("movupd (%%ecx), %%xmm2\n" : : "c" (a_mask));
	while (img < max && !not_empty)
	{
		asm __volatile__
		(
			"movapd (%%esi), %%xmm0\n"
			"pand %%xmm2, %%xmm0\n"
			"psadbw %%xmm1, %%xmm0\n"
			"movd %%xmm0, (%%edx)\n"
			: "=d" (ne_ptr)
			: "S" (img), "d" (ne_ptr)
		);
		img += 16;
	}

	if (not_empty)
		return 0;

	return 1;
}

void zero_transparent_sse2 (stream_info_t *s_info, char *img)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;
	char zero[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	char a_mask[16] = {0,0,0,0xff,0,0,0,0xff,0,0,0,0xff,0,0,0,0xff};
	char full[16] = {0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff};

	asm __volatile__ ("movupd (%%eax), %%xmm1\n" : : "a" (zero));
	asm __volatile__ ("movupd (%%ecx), %%xmm2\n" : : "c" (a_mask));
	asm __volatile__ ("movupd (%%edx), %%xmm3\n" : : "d" (full));
	while (img < max)
	{
		asm __volatile__
		(
			"movapd (%%eax), %%xmm0\n"
			"movapd %%xmm0, %%xmm5\n"
			"pand %%xmm2, %%xmm5\n"
			"pcmpeqd %%xmm1, %%xmm5\n"
			"pxor %%xmm3, %%xmm5\n"
			"pand %%xmm5, %%xmm0\n"
			"movapd %%xmm0, (%%eax)\n"
			: "=a" (img)
			: "a" (img)
		);
		img += 16;
	}
}

void swap_rb_sse2 (stream_info_t *s_info, char *img, char *out)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;
	char ag_mask[16] = {0,0xff,0,0xff,0,0xff,0,0xff,0,0xff,0,0xff,0,0xff,0,0xff};
	char rb_mask[16] = {0xff,0,0xff,0,0xff,0,0xff,0,0xff,0,0xff,0,0xff,0,0xff,0};

	/* Set up masks */
	asm volatile
	(
		"movdqa (%%eax), %%xmm0\n"
		"movdqa (%%ecx), %%xmm5\n"
		:
		: "a" (ag_mask), "c" (rb_mask)
	);

	/* Convert */
	while (img < max)
	{
		asm volatile
		(
			"movdqa (%%esi), %%xmm1\n"
			"movdqa %%xmm1, %%xmm2\n"
			"movdqa %%xmm1, %%xmm7\n"
			"psrld $16, %%xmm2\n"
			"pslld $16, %%xmm1\n"
			"por %%xmm2, %%xmm1\n"
			"pand %%xmm0, %%xmm7\n"
			"pand %%xmm5, %%xmm1\n"
			"por %%xmm7, %%xmm1\n"
			"movdqa %%xmm1, (%%edi)\n"
			: "=D" (out)
			: "S" (img), "D" (out)
		);
		img += 16;
		out += 16;
	}
}

int is_identical_c (stream_info_t *s_info, char *img, char *img_old)
{
	uint32_t *max = (uint32_t *)(img + s_info->i_width * s_info->i_height * 4);
	uint32_t *im = (uint32_t *)img;
	uint32_t *im_old = (uint32_t *)img_old;

	while (im < max)
	{
		if (!((char *)im)[3])
			*im = 0;
		if (*(im++) ^ *(im_old++))
			return 0;
	}

	return 1;
}

int is_empty_c (stream_info_t *s_info, char *img)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;
	char *im = img;

	while (im < max)
	{
		if (im[3])
			return 0;
		im += 4;
	}

	return 1;
}

void zero_transparent_c (stream_info_t *s_info, char *img)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;
	char *im = img;

	while (im < max)
	{
		if (!im[3])
			*(uint32_t *)img = 0;
		im += 4;
	}
}

void swap_rb_c (stream_info_t *s_info, char *img, char *out)
{
	char *max = img + s_info->i_width * s_info->i_height * 4;

	while (img < max)
	{
		out[0] = img[2];
		out[1] = img[1];
		out[2] = img[0];
		out[3] = img[3];
		img += 4;
		out += 4;
	}
}

int detect_sse2 ()
{
	static int detection = -1;
	unsigned int func = 0x00000001;
	unsigned int eax, ebx, ecx, edx;

	if (detection != -1)
		return detection;

	asm volatile
	(
		"cpuid\n"
		: "=a" (eax), "=b" (ebx), "=c" (ecx), "=d" (edx)
		: "a" (func)
	);

	/* SSE2:  edx & 0x04000000
	 * SSSE3: ecx & 0x00000200
	 */
	detection = (edx & 0x04000000) ? 1 : 0;

	if (detection)
		fprintf(stderr, "CPU: Using SSE2 optimized functions.\n");
	else
		fprintf(stderr, "CPU: Using pure C functions.\n");

	return detection;
}

int is_identical (stream_info_t *s_info, char *img, char *img_old)
{
	if (detect_sse2())
		return is_identical_sse2(s_info, img, img_old);
	else
		return is_identical_c(s_info, img, img_old);
}

int is_empty (stream_info_t *s_info, char *img)
{
	if (detect_sse2())
		return is_empty_sse2(s_info, img);
	else
		return is_empty_c(s_info, img);
}

void zero_transparent (stream_info_t *s_info, char *img)
{
	if (detect_sse2())
		zero_transparent_sse2(s_info, img);
	else
		zero_transparent_c(s_info, img);
}

void swap_rb (stream_info_t *s_info, char *img, char *out)
{
	if (detect_sse2())
		swap_rb_sse2(s_info, img, out);
	else
		swap_rb_c(s_info, img, out);
}

/* SMPTE non-drop time code */
void mk_timecode (int frame, int fps, char *buf) /* buf must have length 12 (incl. trailing \0) */
{
	int frames, s, m, h;
	int tc = frame;

	tc = frame;
	frames = tc % fps;
	tc /= fps;
	s = tc % 60;
	tc /= 60;
	m = tc % 60;
	tc /= 60;
	h = tc;

	if (h > 99)
	{
		fprintf(stderr, "Timecodes above 99:59:59:99 not supported: %Lu:%02u:%02u:%02u\n", h, m, s, frames);
		exit(1);
	}

	if (snprintf(buf, 12, "%02d:%02d:%02d:%02d", h, m, s, frames) != 11)
	{
		fprintf(stderr, "Timecode lead to invalid format: %s\n", buf);
		exit(1);
	}
}

void print_usage ()
{
	fprintf(stderr, "Usage: avs2bdnxml INPUT TRACKNAME LANGUAGE VIDEOFORMAT FRAMERATE OUTPUT\n\n"
		"INPUT\t\tInput file (AviSynth script)\n"
		"TRACKNAME\tName of track, like: Undefined\n"
		"LANGUAGE\tLanguage code, like: und\n"
		"VIDEOFORMAT\tEither of: 480i, 480p, 576i, 720p, 1080i, 1080p\n"
		"FRAMERATE\tEither of: 23.976, 24, 25, 29.97, 50, 59.94\n"
		/*"         \tFor drop timecodes, use: 23.976d, 29.97d, 59.94d\n"*/
		"OUTPUT\t\tOutput file in BDN XML format\n\n"
		"Example: avs2bdnxml input.avs Undefined und 1080p 23.976 output.xml\n");
}

struct event_list_node_s
{
	int start_frame;
	int end_frame;
	struct event_list_node_s *next;
};

typedef struct
{
	struct event_list_node_s *start;
	struct event_list_node_s *end;
} event_list_head_t;

void add_event_xml(event_list_head_t *events, int start, int end)
{
	struct event_list_node_s *new = calloc(1, sizeof(struct event_list_node_s));
	new->start_frame = start;
	new->end_frame = end;
	if (events->start == NULL)
	{
		events->start = new;
		events->end = new;
	}
	else
	{
		(events->end)->next = new;
		events->end = new;
	}
}

struct framerate_entry_s
{
	char *name;
	char *out_name;
	int rate;
	int drop;
};

/* Most of the time seems to be spent in AviSynth (about 4/5). */
int main (int argc, char *argv[])
{
	struct framerate_entry_s framerates[] = { {"23.976", "23.976", 24, 0}
	                                        /*, {"23.976d", "23.976", 24000/1001.0, 1}*/
	                                        , {"24", "24", 24, 0}
	                                        , {"25", "25", 25, 0}
	                                        , {"29.97", "29.97", 30, 0}
	                                        /*, {"29.97d", "29.97", 30000/1001.0, 1}*/
	                                        , {"50", "50", 50, 0}
	                                        , {"59.94", "59.94", 60, 0}
	                                        /*, {"59.94d", "59.94", 60000/1001.0, 1}*/
	                                        , {NULL, NULL, 0, 0}
	                                        };
	char *avs_filename, *track_name, *language, *video_format, *frame_rate, *out_filename;
	char *in_img, *old_img, *tmp, *out_buf;
	char *intc_buf, *outtc_buf;
	char *drop_frame;
	int fps;
	int frames;
	int first_frame = -1, start_frame, end_frame;
	int num_of_events = 0;
	int i;
	int have_line = 0;
	int must_zero;
	int checked_empty;
	int auto_cut = 0;
	int progress_step = 1000;
	int bench_start = time(NULL);
	avis_input_t *avis_hnd;
	stream_info_t *s_info = malloc(sizeof(stream_info_t));
	event_list_head_t *events = malloc(sizeof(event_list_head_t));
	struct event_list_node_s *node;
	FILE *fh;

	/* Get args */
	if (argc < 7)
	{
		print_usage();
		return 0;
	}
	avs_filename = argv[1];
	track_name = argv[2];
	language = argv[3];
	video_format = argv[4];
	out_filename = argv[6];

	/* TODO: Sanity check video_format and frame_rate. */

	/* Get frame rate */
	frame_rate = NULL;
	i = 0;
	while (framerates[i].name != NULL)
	{
		if (!strcasecmp(framerates[i].name, argv[5]))
		{
			frame_rate = framerates[i].out_name;
			fps = framerates[i].rate;
			drop_frame = framerates[i].drop ? "true" : "false";
		}
		i++;
	}
	if (frame_rate == NULL)
	{
		fprintf(stderr, "Error: Invalid framerate (%s).\n", argv[5]);
		return 1;
	}

	/* Detect CPU features */
	detect_sse2();

	/* Get video info and allocate buffer */
	if (open_file_avis(avs_filename, &avis_hnd, s_info))
	{
		print_usage();
		return 1;
	}
	in_img  = calloc(s_info->i_width * s_info->i_height * 4 + 16 * 2, sizeof(char)); /* allocate + 16 for alignment, and + n * 16 for over read/write */
	old_img = calloc(s_info->i_width * s_info->i_height * 4 + 16 * 2, sizeof(char)); /* see above */
	out_buf = calloc(s_info->i_width * s_info->i_height * 4 + 16 * 2, sizeof(char));

	/* align buffers */
	in_img  = in_img + (16 - ((int)in_img % 16));
	old_img = old_img + (16 - ((int)old_img % 16));
	out_buf = out_buf + (16 - ((int)out_buf % 16));

	/* get frame number */
	frames  = get_frame_total_avis(avis_hnd);

	/* No frames mean nothing to do */
	if (frames < 1)
	{
		fprintf(stderr, "No frames found.\n");
		return 0;
	}

	/* Set progress step */
	if (frames < 1000)
	{
		if (frames > 200)
			progress_step = 50;
		else if (frames > 50)
			progress_step = 10;
		else
			progress_step = 1;
	}

	/* Initialize events */
	events->start = NULL;
	events->end = NULL;

	if (detect_sse2())
		asm ("emms\n");

	/* Process frames */
	for (i = 0; i < frames; i++)
	{
		if (read_frame_avis(in_img, avis_hnd, i))
		{
			fprintf(stderr, "Error reading frame.\n");
			return 1;
		}
		checked_empty = 0;

		/* Progress indicator */
		if (i % (frames / progress_step) == 0)
		{
			fprintf(stderr, "\rProgress: %d/%d - Lines: %d", i, frames, num_of_events);
		}

		/* If we are outside any lines, check for empty frames first */
		if (!have_line)
		{
			if (is_empty(s_info, in_img))
				continue;
			else
				checked_empty = 1;
		}

		/* Check for duplicate, unless first frame */
		if (i && have_line && is_identical(s_info, in_img, old_img))
			continue;
		/* Mark frames that were not used as new image in comparison to have transparent pixels zeroed */
		else if (!(i && have_line))
			must_zero = 1;

		/* Not a dup, write end-of-line, if we had a line before */
		if (have_line)
		{
			add_event_xml(events, start_frame, i);
			end_frame = i;
			have_line = 0;
		}

		/* Check for empty frame, if we didn't before */
		if (!checked_empty && is_empty(s_info, in_img))
			continue;

		/* Zero transparent pixels, if needed */
		if (must_zero)
			zero_transparent(s_info, in_img);
		must_zero = 0;

		/* Not an empty frame, start line */
		have_line = 1;
		start_frame = i;
		swap_rb(s_info, in_img, out_buf);
		write_png(start_frame, out_buf, s_info->i_width, s_info->i_height);
		num_of_events++;
		if (first_frame == -1)
			first_frame = i;

		/* Save image for next comparison. */
		tmp = in_img;
		in_img = old_img;
		old_img = tmp;
	}

	fprintf(stderr, "\rProgress: %d/%d - Lines: %d - Done\n", i, frames, num_of_events);

	/* Add last event, if available */
	if (have_line)
	{
		add_event_xml(events, start_frame, i - 1);
		auto_cut = 1;
		end_frame = i - 1;
	}

	/* Check if we actually have any events */
	if (first_frame == -1)
	{
		fprintf(stderr, "No events detected. Refusing to write XML file.\n");
		return 1;
	}

	if (detect_sse2())
		asm ("emms\n");

	/* Initialize timecode buffers */
	intc_buf = calloc(12, 1);
	outtc_buf = calloc(12, 1);

	/* Creating XML file */
	if ((fh = fopen(out_filename, "w")) == 0)
	{
		perror("Error opening output file");
		return 1;
	}

	/* Write XML header */
	mk_timecode(first_frame, fps, intc_buf);
	mk_timecode(end_frame, fps, outtc_buf);
	fprintf(fh, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<BDN Version=\"0.93\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
		"xsi:noNamespaceSchemaLocation=\"BD-03-006-0093b BDN File Format.xsd\">\n"
		"<Description>\n"
		"<Name Title=\"%s\" Content=\"\"/>\n"
		"<Language Code=\"%s\"/>\n"
		"<Format VideoFormat=\"%s\" FrameRate=\"%s\" DropFrame=\"%s\"/>\n"
		"<Events LastEventOutTC=\"%s\" FirstEventInTC=\"%s\"\n", track_name, language, video_format, frame_rate, drop_frame, outtc_buf, intc_buf);

	mk_timecode(0, fps, intc_buf);
	mk_timecode(frames, fps, outtc_buf);
	fprintf(fh, "ContentInTC=\"%s\" ContentOutTC=\"%s\" NumberofEvents=\"%d\" Type=\"Graphic\"/>\n"
		"</Description>\n"
		"<Events>\n", intc_buf, outtc_buf, num_of_events);

	/* Write XML events */
	if (events->start != NULL)
	{
		node = events->start;
		do
		{
			mk_timecode(node->start_frame, fps, intc_buf);
			mk_timecode(node->end_frame, fps, outtc_buf);
			
			if (auto_cut && node->end_frame == frames - 1)
			{
				mk_timecode(node->end_frame + 1, fps, outtc_buf);
			}
			
			fprintf(fh, "<Event Forced=\"False\" InTC=\"%s\" OutTC=\"%s\">\n<Graphic Width=\"%d\" Height=\"%d\" X=\"0\" Y=\"0\">%08d.png</Graphic>\n</Event>\n", intc_buf, outtc_buf, s_info->i_width, s_info->i_height, node->start_frame);
			node = node->next;
		}
		while (node != NULL);
	}

	/* Write XML footer */
	fprintf(fh, "</Events>\n</BDN>\n");

	/* Cleanup */
	fclose(fh);
	close_file_avis(avis_hnd);

	/* Give runtime */
	/*fprintf(stderr, "Time elapsed: %d\n", time(NULL) - bench_start);*/

	return 0;
}
