/*----------------------------------------------------------------------------
 * avs2bdnxml - Generates BluRay subtitle stuff from RGBA AviSynth scripts
 * Copyright (C) 2008-2009 Arne Bochem <avs2bdnxml at ps-auxw de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *----------------------------------------------------------------------------*/

#include <stdint.h>
#include <string.h>
#include <limits.h>
#include "auto_split.h"

/* Transparent pixels are assumed to be set to zero */

void auto_crop (pic_t p, crop_t *c)
{
	uint32_t *b = (uint32_t *)p.b;
	int min_x = INT_MAX, max_x = INT_MIN, min_y = INT_MAX, max_y = INT_MIN;
	int seen_pixel = 0;
	int x, y;

	for (y = c->y; y < c->y + c->h && y < p.h; y++)
		for (x = c->x; x < c->x + c->w && x < p.w; x++)
			if (b[x + p.s * y])
			{
				seen_pixel = 1;
				if (x < min_x)
					min_x = x;
				if (x > max_x)
					max_x = x;
				if (y < min_y)
					min_y = y;
				if (y > max_y)
					max_y = y;
			}

	if (!seen_pixel)
	{
		c->w = 0;
		c->h = 0;
	}
	else
	{
		c->x = min_x;
		c->y = min_y;
		c->w = max_x - min_x + 1;
		c->h = max_y - min_y + 1;
	}
}

static int block_state (pic_t p, crop_t c)
{
	uint32_t *b = (uint32_t *)p.b;
	int x, y;

	for (y = c.y; y < c.y + c.h && y < p.h; y++)
		for (x = c.x; x < c.x + c.w && x < p.w; x++)
			if (b[x + p.s * y])
				return -1;

	return 0;
}

#define GRID_BLOCKS 10
typedef crop_t rect_t;

static int line_ok (int grid[GRID_BLOCKS + 1][GRID_BLOCKS + 1], int x, int y, int w)
{
	int i;

	if (grid[y][MAX(0, x - 1)] == -1)
		return 0;

	for (i = 0; i < w; i++)
		if (grid[y][x + i] != -1)
			return 0;

	if (grid[y][MIN(GRID_BLOCKS, x + i)] == -1)
		return 0;

	return 1;
}

static void set_line (int grid[GRID_BLOCKS + 1][GRID_BLOCKS + 1], int x, int y, int w, int n)
{
	int i;

	for (i = 0; i < w; i++)
		grid[y][x + i] = n;
}

static rect_t make_rect (int grid[GRID_BLOCKS + 1][GRID_BLOCKS + 1], int x, int y, int n)
{
	rect_t r = {x, y, 1, 1};
	int line_length = 1;

	/* Get length of first rectangle line, and assign rect number */
	grid[y][x] = n;
	while (line_length + x < GRID_BLOCKS + 1 && grid[y][x + line_length] == -1)
	{
		grid[y][x + line_length] = n;
		line_length++;
	}
	r.w = line_length;

	/* Add lines while available */
	while (line_ok(grid, x, r.y + r.h, r.w))
		set_line(grid, x, r.y + r.h++, r.w, n);

	return r;
}

static rect_t merge_rects (rect_t r1, rect_t r2)
{
	rect_t r;

	/* Set rectangle that covers both input rectangles */
	r.x = MIN(r1.x, r2.x);
	r.y = MIN(r1.y, r2.y);
	r.w = MAX(r1.x + r1.w, r2.x + r2.w) - r.x;
	r.h = MAX(r1.y + r1.h, r2.y + r2.h) - r.y;

	return r;
}

static int score_rect (rect_t r)
{
	return r.w * r.h;
}

/* crop_t *c - Array of length 2 */
int auto_split (pic_t p, crop_t *c)
{
	crop_t c1 = {0, 0, 0, 0};
	crop_t c2 = {0, 0, 0, 0};
	crop_t t;
	rect_t rects[(GRID_BLOCKS + 1) * (GRID_BLOCKS + 1)];
	rect_t r1, r2, rt1, rt2;
	int grid[GRID_BLOCKS + 1][GRID_BLOCKS + 1];
	int score_t1, score_t2, score_r1 = 0, score_r2 = 0;
	int n_rect = 0;
	int score = 0;
	int bw, bh;
	int x, y;
	int i, j;

	memset(grid, 0, sizeof(int) * (GRID_BLOCKS + 1) * (GRID_BLOCKS + 1));
	bw = p.w / GRID_BLOCKS;
	bh = p.h / GRID_BLOCKS;
	t.w = bw;
	t.h = bh;

	/* Determine state of blocks */
	for (y = 0; (t.y = y * bh) < p.h; y++)
		for (x = 0; (t.x = x * bw) < p.w; x++)
			grid[y][x] = block_state(p, t);

	/* Create rectangles */
	for (y = 0; y < GRID_BLOCKS + 1; y++)
		for (x = 0; x < GRID_BLOCKS + 1; x++)
			if (grid[y][x] == -1)
			{
				rects[n_rect] = make_rect(grid, x, y, n_rect);
				n_rect++;
			}

	/* Shouldn't happen, empty frame */
	if (!n_rect)
	{
		c[0] = c1;
		c[1] = c2;
		return 0;
	}

	/* Single rectangle */
	if (n_rect == 1)
	{
		c1.x = rects[0].x * bw;
		c1.y = rects[0].y * bh;
		c1.w = rects[0].w * bw;
		c1.h = rects[0].h * bh;
		auto_crop(p, &c1);
		c[0] = c1;
		c[1] = c2;
		return 1;
	}

	/* Any other number of rectangles, first find most "distant" ones */
	for (i = 0; i < n_rect; i++)
		for (j = 0; j < n_rect; j++)
			if (i == j)
				continue;
			else
			{
				rt1 = merge_rects(rects[i], rects[j]);
				score_t1 = score_rect(rt1);
				if (score <= score_t1)
				{
					score = score_t1;
					r1 = rects[i];
					r2 = rects[j];
					score_r1 = score_rect(r1);
					score_r2 = score_rect(r2);
				}
			}

	/* Merge all other rectangles with the "nearest" one */
	for (i = 0; i < n_rect; i++)
	{
		rt1 = merge_rects(r1, rects[i]);
		rt2 = merge_rects(r2, rects[i]);
		score_t1 = score_rect(rt1);
		score_t2 = score_rect(rt2);
		if (score_t1 - score_r1 < score_t2 - score_r2)
		{
			r1 = rt1;
			score_r1 = score_t1;
		}
		else
		{
			r2 = rt2;
			score_r2 = score_t2;
		}
	}

	/* Turn rectangles into crops */
	c1.x = r1.x * bw;
	c1.y = r1.y * bh;
	c1.w = r1.w * bw;
	c1.h = r1.h * bh;
	c2.x = r2.x * bw;
	c2.y = r2.y * bh;
	c2.w = r2.w * bw;
	c2.h = r2.h * bh;

	/* Minimize surface and return them */
	auto_crop(p, &c1);
	auto_crop(p, &c2);
	c[0] = c1;
	c[1] = c2;
	return 2;
}

